classdef vct3
    % classdef vct3 - implements class for cartesian 3 vectors, intended to
    % be used with classes RotMX and Frame.
    %
    % Copyright (C) Russell H. Taylor, Johns Hopkins University, 2013
    %
    % Properties
    %   el(3,N) is a 3xN matrix of column vectors, each corresponding to a
    %           cartesian vector
    %
    % Notation 
    %   N                           number of column vectors
    %   v, v1, v2, ...              vct3 objects
    %   s, x,y,z,a,b                scalar objects
    %   M, M1, M2, ...              Matrix object 
    %   X                           May be matrix or scalar (see v1.*v2)
    %   k                           Integer
    %   S                           Subscript object
    %
    % Constructors
    %   v = vct3()                  single vct3 
    %   v = vct3(N)                 matrix of N vct3 objects
    %   v = vct3(v)                 copy constructor
    %   v = vct3(M)                 creates vct3 from 3xN matrix M
    %   v = vct3(x,y,z)             equivalent to vct3([x,y,z]')
    %   v = unit(v1)                returns v/norm(v)
    %
    %   v = vct3.null()             [0,0,0]
    %   v = vct3.rand(a,b)          returns random vector between a and b
    %   v = vct3.rand()             equivalent to vct3.rand(0,1)
    %
    % Indexing and access methods
    %   k = NumEl(v)                returns N
    %   s = v(k)                    returns v1.el(k,1) if N=1
    %                               returns vct3(v1.el(:,k)) otherwise
    %   v(k) = X                    if N == 1, sets k'th element to X
    %                               otherwise, sets k'th columns to X
    %                               returns v
    %   M = X(v1)                   computes v1.el(1,:)
    %   M = Y(v1)                   computes v1.el(2,:)
    %   M = Z(v1)                   computes v1.el(3,:) 
    %
    % Computational methods
    %
    %   v = v1+v2                   computes v1+v2
    %   v = v1-v2                   computes v1-v2
    %   v = -v1                     computes -v1
    %   v = v1*v2                   computes v1*v2 = v1.el'*v2.el
    %   v = v1*s                    computes v1*s  = vct3(v1*s)
    %   X = v1*M                    computes v1*M  = v1.el'*M
    %   v = v1.*v2                  computes v1.*v2= vct3(v1.el*v2.el)
    %   v = v1.*X                   computes v1.*v2= vct3(v1.el*X)
    %   v = v1/s                    computes v1./s = vct3(v1.el/s)
    %   v = v1./s                   computes v1./s = vct3(v1.el./s)
    %   
    %   v = plus(v1,v2)             computes v1+v2
    %   v = minus(v1,v2)            computes v1-v2
    %   v = uminus(v1)              computes -v1
    %   M = mtimes(v1,v2)           computes v1*v2 = v1.el'*v2.el
    %   v = mtimes(v1,s)            computes v1*s  = vct3(v1*s)
    %   X = mtimes(v1,M)            computes v1*M  = v1.el'*M
    %   v = times(v1,v2)            computes v1.*v2= vct3(v1.el*v2.el)
    %   v = times(v1,X)             computes v1.*s = vct3(v1.el.*X)
    %   v = rdivide(v1,s)           computes v1./s = vct3(v1.el./s)
    %
    %   v = cross(v1,v2)            computes vector cross product
    %
    %   v = abs(v)                  computes vct3(abs(v1.el))
    %   v = mean(v1)                computes vct3(mean(v1.el,2))
    %   s = norm(v1,p)              computes norm(v1.el,p)
    %
    % Other methods
    %   display(v)                  console display (calls disp)
    %   disp(v)                     basic console display
    % 
    % Copyright (C) Russell H. Taylor 2013
    % For use with CIS I only
    % Do not redistribute without written permission from Russell Taylor  
    
    properties
        el = [0 0 0]';
    end
    
    methods
        function v = vct3(varargin)
            switch nargin
                case 0
                    return; % default is null vct3
                case 1
                    if isa(varargin{1},'vct3')
                        v.el = varargin{1}.el;
                    elseif isscalar(varargin{1})
                        v.el = zeros(3,varargin{1});
                    else 
                        sz = size(varargin{1});
                        if sz(1) == 3 && sz(2) == 1
                            v.el = varargin{1};
                        elseif sz(2)==3 && sz(1) == 1
                            v.el = varargin{1}';
                        else
                            error('BadArg','Bad argument to vct3',varargin);
                        end
                    end
                    return;
                case 3
                    v.el = [varargin{1},varargin{2},varargin{3}]';  % assume x,y,z
                otherwise
                    error('BadArg','Wrong number of arguments to vct3',nargin);
            end
        end
        
        function v = subsref(v1,S)
           switch S(1).type 
                case '()'
                    if size(v1)==[1 1]
                        v = subsref(v1.el,S);
                    else
                        v = builtin('subsref',v1,S);
                    end
                otherwise
                    v=builtin('subsref',v1,S);
            end
        end
            
        function v = subsasgn(v,S,X)
           switch S(1).type
               case '()'
                   if isa(X,'vct3')
                       if size(v)==[1 1]
                           SS = substruct('.','el','()',{':',S.subs{:}});
                           v = builtin('subsasgn',v,SS,X.el);
                       else
                           v = builtin('subsasgn',v,S,X);
                       end
                   elseif isscalar(X) || size(X,1)==1 
                       SS = substruct('.','el','()',{S.subs{:},[1]});
                       v = builtin('subsasgn',v,SS,X);                       
                   else
                       error('BadArg','Bad right hand side value for subsasgn',X);
                   end
               otherwise
                   v = builtin('subsasgn',v,S,X);
           end
        end    
        
        function s = X(v)
            s = v.el(1,:);
        end
       
        function s = Y(v)
            s = v.el(2,:);
        end
        
               
        function s = Z(v)
            s = v.el(3,:);
        end
        
        function s = mtimes(v1,v2)
            if isa(v2,'vct3')||isa(v2,'vct3Array')
                s = v1.el'*v2.el; 
            elseif isscalar(v2)
                s = vct3(v1.el*v2);
            else
               s = v1.el'*v2;
            end;                     
        end
        
        function v3 = times(v1,v2)
             if isa(v2,'vct3')
                v3 = vct3(v1.el.*v2.el);
             elseif isa(v2,'vct3Array')
                 v3 = vct3Array(v1.el.*v2.el);
             else
                v3 =vct3(v1.el.*v2);
             end;
        end
        
        function v3 = uminus(v1)
            v3 = vct3(-v1.el);
        end
        
        function v = abs(v1)
            v = vct3(abs(v1.el));
        end
        
        function v3 = plus(v1,v2)
            switch class(v2)
                case 'vct3Array'
                    v3 = vct3Array(bsxfun(@plus,v1.el,v2.el));
                case 'vct3'
                    v3 = vct3(bsxfun(@plus,v1.el,v2.el));
                otherwise 
                    v3 = vct3(bsxfun(@plus,v1.el,v2));
                end;
        end
        
        function v3 = minus(v1,v2)
            switch class(v2)
                case 'vct3Array'
                    v3 = vct3Array(bsxfun(@minus,v1.el,v2.el));
                case 'vct3'
                    v3 = vct3(bsxfun(@minus,v1.el,v2.el));
                otherwise 
                    v3 = vct3(bsxfun(@minus,v1.el,v2));
                end;
        end
          
        function v = rdivide(v1,s)
            v = vct3(v1.el/s);
        end
        
        function result = mean(v)
            result = mean(v.el,2);
        end
        
        function v = unit(v1)
            v = vct3(v1.el/norm(v1.el));
        end
        
        function s = norm(varargin)
            switch nargin
                case 1
                    s = norm(varargin{1}.el,2);
                    return;
                case 2
                    s = norm(varargin{1}.el,varargin{2});
                    return;
                otherwise
                    error('BadArg','Wrong number of arguments to vct3.norm',nargin);
            end
        end
         
        function v=cross(v1,v2)
            v = vct3(cross(v1.el,v2.el));
        end
        
        function s=dot(v1,v2)
            s = dot(v1.el,v2.el);
        end
        
        function disp(v)
            sz=size(v);
            if sz==[1,1]
                disp(v.el);
            else
                for i=1:sz(1)
                    el =[];
                    for j=1:sz(2)
                        el=[el,v(i,j).el];
                    end
                    disp(el);
                end
            end          
        end
        
        function display(v)          
            if isequal(get(0,'FormatSpacing'),'compact')
               disp([inputname(1) ' =']);
               disp(v);
            else
               disp(' ')
               disp([inputname(1) ' =']);
               disp(' ');
               disp(v);
            end
        end
               
    end
    
    methods (Static)
        function v = null()
            v = vct3(0,0,0);
        end
        
        function v = rand(a,b)
            switch nargin
                case 0
                    v = vct3(rand(3,1)); 
                case 1
                    if isa(a,'vct3')
                        ab = 2*a.el;
                        v = vct3(-a.el+ab.*rand(3,1));
                    else
                        v = vct3(-a +2*a.*rand(3,1));
                    end
                case 2
                   if isa(a,'vct3')
                       ab = b.el-a.el;
                       v= vct3(a.el+ab.*rand(3,1));
                   else
                       v = vct3(a +(b-a).*rand(3,1));
                   end
                otherwise
                    error('BadArg','Wrong number of arguments to vct3.rand',nargin);
            end
       end
   end
    
end

